#ifndef SHORTEST_PATH_HPP_INCLUDED
#define SHORTEST_PATH_HPP_INCLUDED

#include "const_nodes.hpp"

#include <stack>

#include <cmath>

class Shortest_path {
  public:
    Shortest_path(const Const_Nodes *nodes,const Const_Nodes *revnodes):_nodes(nodes),_revnodes(revnodes) {;}

    void print_shortest_stats(std::ostream &out,size_t cutoff = 20) {
      out<<"Shortest paths by lenght:"<<std::endl;
      for(size_t i = 0; i < _nb_bydist.size(); ++i) {
        out<<i<<": "<<_nb_bydist[i]<<std::endl;
      }
      // find which
      if (_nb_bydist.size() < cutoff + 1) return;
      else {
        size_t count = 0;
        for (size_t i = cutoff; i < _nb_bydist.size();++i) {
          count += _nb_bydist[i];
        }
        if (count > 200) return;
      }
      std::vector<std::vector<size_t>> longlist;
      longlist.resize(_nb_bydist.size() - cutoff);
      for (size_t i = 0; i < _distances.size();++i) {
        if (_distances[i] >= cutoff) {
          longlist[_distances[i] - cutoff].push_back(i);
        }
      }
      out<<"Longest path:"<<std::endl;
      for (size_t i = 0; i < longlist[_nb_bydist.size() - cutoff - 1].size(); ++i) {
        std::vector<int> path;
        path.push_back(longlist[_nb_bydist.size() - cutoff - 1][i]);
        find_long_path_from(path, _nb_bydist.size() - 1);
        if (path.size() > 1) {
          for (size_t j = 0; j < path.size() - 1; ++j) {
            out<<_nodes->id(path[j])<<" -> ";
          }
          out<<_nodes->id(path.back())<<std::endl;
        }
      }
      out<<"Far articles: "<<std::endl;
      for (int i = _nb_bydist.size() - 1; i >= cutoff; --i) {
        out<<i<<":\n";
        for (size_t j = 0; j < longlist[i - cutoff].size() -1; ++j) {
          out<<_nodes->id(longlist[i-cutoff][j])<<", ";
        }
        out<<_nodes->id(longlist[i-cutoff].back())<<std::endl;
      }
    }

    void list_distances(std::ostream &out, int dist) {
      for (size_t i = 0; i < _distances.size(); ++i) {
        if (_distances[i] == dist) {
          out<<_nodes->id(i)<<std::endl;
        }
      }
    }

    void print_center_stats(std::ostream &out) {
      out<<"Eccentricity: "<<_eccen<<std::endl;
      out<<"Number of centers: "<<_centers.size()<<std::endl;
      out<<"Centers:\n";
      for (size_t i = 0; i < _centers.size(); ++i) {
        out<<_nodes->id(_centers[i])<<std::endl;
      }
    }
     
    int find_distance(size_t from, size_t to) {
      shortest_from(to);
      return _distances[from];
    }

    void sample_stats(std::ostream &out, size_t nb_ech) {
      std::vector<int> max_dist, mid_dist,mid_size, sp_list, outer_list, highway_orphan;
      std::vector<std::vector<int>> max_dist_nb(50);
      max_dist.reserve(nb_ech);
      mid_dist.reserve(nb_ech);
      mid_size.reserve(nb_ech);
      size_t step_size = _nodes->size()/nb_ech;
      if (step_size < 1) {
        step_size = 1;
        nb_ech = _nodes->size();
      }
      for (size_t i = 0; i < nb_ech; ++i) {
        out<<"\r"<<i<<"       "<<std::flush;
        try {shortest_from(i*step_size);}
        catch (std::runtime_error) {
          highway_orphan.push_back(i*step_size);
          continue;
        }
        if (_nb_bydist.size() == 1) {
          continue;
        }
        if (_nb_bydist.size() > 20) {
          outer_list.push_back(i*step_size);
        }
        max_dist.push_back(_nb_bydist.size()-1);
        if (_nb_bydist.size() <= 50) {
          max_dist_nb[_nb_bydist.size()-1].push_back(_nb_bydist.back());
        }
        // find max
        size_t old = _nb_bydist[0];
        bool increasing = true;
        for (size_t j = 1; j < _nb_bydist.size(); ++j) {
          if (_nb_bydist[j] >= old) {
            if (not increasing) {
              sp_list.push_back(i*step_size);
              increasing = true;
            }
          } else {
            if (increasing) {
              mid_dist.push_back(j-1);
              mid_size.push_back(_nb_bydist[j-1]);
              increasing = false;
            }
          }
          old = _nb_bydist[j];
        }
      } // gather samples
      out<<"\r";
      double mean, stdev;
      int min, max;
      min = max_dist[0]; max = max_dist[0]; mean = 0; stdev = 0;
      for (size_t i = 0; i < max_dist.size(); ++i) {
        max = std::max(max,max_dist[i]);
        min = std::min(min,max_dist[i]);
        mean += max_dist[i];
      }
      mean /= max_dist.size();
      for (size_t i = 0; i < max_dist.size(); ++i) {
        stdev += (max_dist[i] - mean)*(max_dist[i] - mean);
      }
      stdev = std::sqrt(stdev/(max_dist.size()-1));
      std::nth_element(max_dist.begin(),          max_dist.begin() + max_dist.size()/4, max_dist.end());
      std::nth_element(max_dist.begin() + max_dist.size()/4 + 1, max_dist.begin() + max_dist.size()/2, max_dist.end());
      std::nth_element(max_dist.begin() + max_dist.size()/2 + 1, max_dist.begin() + max_dist.size()*3/4, max_dist.end());
      out<<"Max distance: min: "<<min<<" max: "<<max<<" mean: "<<mean<<" stdev: "<<stdev
         <<" First quartile: "<<max_dist[max_dist.size()/4]<<" second: "<<max_dist[max_dist.size()/2]<< " third: "<<max_dist[max_dist.size()*3/4]<<std::endl;

      min = mid_dist[0]; max = mid_dist[0]; mean = 0; stdev = 0;
      for (size_t i = 0; i < mid_dist.size(); ++i) {
        max = std::max(max,mid_dist[i]);
        min = std::min(min,mid_dist[i]);
        mean += mid_dist[i];
      }
      mean /= mid_dist.size();
      for (size_t i = 0; i < mid_dist.size(); ++i) {
        stdev += (mid_dist[i] - mean)*(mid_dist[i] - mean);
      }
      stdev = std::sqrt(stdev/(mid_dist.size()-1));
      std::nth_element(mid_dist.begin(),          mid_dist.begin() + mid_dist.size()/4, mid_dist.end());
      std::nth_element(mid_dist.begin() + mid_dist.size()/4 + 1, mid_dist.begin() + mid_dist.size()/2, mid_dist.end());
      std::nth_element(mid_dist.begin() + mid_dist.size()/2 + 1, mid_dist.begin() + mid_dist.size()*3/4, mid_dist.end());
      out<<"Mid distance: min: "<<min<<" max: "<<max<<" mean: "<<mean<<" stdev: "<<stdev
         <<" First quartile: "<<mid_dist[mid_dist.size()/4]<<" second: "<<mid_dist[mid_dist.size()/2]<< " third: "<<mid_dist[mid_dist.size()*3/4]<<std::endl;

      min = mid_size[0]; max = mid_size[0]; mean = 0; stdev = 0;
      for (size_t i = 0; i < mid_size.size(); ++i) {
        max = std::max(max,mid_size[i]);
        min = std::min(min,mid_size[i]);
        mean += mid_size[i];
      }
      mean /= mid_size.size();
      for (size_t i = 0; i < mid_size.size(); ++i) {
        stdev += (mid_size[i] - mean)*(mid_size[i] - mean);
      }
      stdev = std::sqrt(stdev/(mid_size.size()-1));
      std::nth_element(mid_size.begin(),          mid_size.begin() + mid_size.size()/4, mid_size.end());
      std::nth_element(mid_size.begin() + mid_size.size()/4 + 1, mid_size.begin() + mid_size.size()/2, mid_size.end());
      std::nth_element(mid_size.begin() + mid_size.size()/2 + 1, mid_size.begin() + mid_size.size()*3/4, mid_size.end());
      out<<"Mid size    : min: "<<min<<" max: "<<max<<" mean: "<<mean<<" stdev: "<<stdev
         <<" First quartile: "<<mid_size[mid_size.size()/4]<<" second: "<<mid_size[mid_size.size()/2]<< " third: "<<mid_size[mid_size.size()*3/4]<<std::endl;

      for (size_t at_i = 0; at_i < max_dist_nb.size(); ++at_i) {
        if (max_dist_nb[at_i].empty()) continue;
        std::vector<int> const & dist_nb = max_dist_nb[at_i];
        if (dist_nb.size() == 1) {
          out<<"Number at "<<at_i<<": "<<dist_nb[0]<<" (hit once)"<<std::endl;
          continue;
        }
        min = dist_nb[0]; max = dist_nb[0]; mean = 0; stdev = 0;
        for (size_t i = 0; i < dist_nb.size(); ++i) {
          max = std::max(max,dist_nb[i]);
          min = std::min(min,dist_nb[i]);
          mean += dist_nb[i];
        }
        mean /= dist_nb.size();
        for (size_t i = 0; i < dist_nb.size(); ++i) {
          stdev += (dist_nb[i] - mean)*(dist_nb[i] - mean);
        }
        stdev = std::sqrt(stdev/(dist_nb.size()-1));
        out<<"Number at "<<at_i<<": min: "<<min<<" max: "<<max<<" mean: "<<mean<<" stdev: "<<stdev<<" size: "<<dist_nb.size()<<std::endl;
      }
      if (sp_list.size() > 0) {
        out<<"Elements with non normal distribution: "<<std::endl;
        for (size_t i = 0; i < sp_list.size(); ++i) {
          out<<_nodes->id(sp_list[i])<<std::endl;
        }
      }
      if (highway_orphan.size() > 0) {
        out<<"Elements disconnected by the removal of excludes: "<<std::endl;
        for (size_t i = 0; i < highway_orphan.size(); ++i) {
          out<<_nodes->id(highway_orphan[i])<<std::endl;
        }
      }
      if (outer_list.size() > 0) {
        out<<"Elements with high eccentricity: "<<std::endl;
        for (size_t i = 0; i < outer_list.size(); ++i) {
          out<<_nodes->id(outer_list[i])<<std::endl;
        }
      }
    }

    void find_bestofchild(std::ostream &out,size_t index) {
      _eccen = _nodes->size();
      _centers.resize(0);
      for (size_t i = 0; i < _nodes->edges_of(index).size();++i) {
        out<<"\r"<<i<<" best:"<<_eccen<<"       "<<std::flush;
        int tmp = shortest_from(_nodes->edges_of(index)[i],_eccen);
        if (tmp < 0) {
          continue;
        } else if (tmp == _eccen) {
          _centers.push_back(_nodes->edges_of(index)[i]);
        } else if (tmp < _eccen) {
          _eccen = tmp;
          _centers.resize(0);
          _centers.push_back(_nodes->edges_of(index)[i]);
        } else {
          throw std::runtime_error("Logic error in shortest_from");
        }
      }
    }   

    void find_center(std::ostream &out) {
      _eccen = _nodes->size();
      _centers.resize(0);
      for (size_t i = 0; i < _nodes->size();++i) {
        out<<"\r"<<i<<" best:"<<_eccen<<"       "<<std::flush;
        int tmp = shortest_from(i,_eccen);
        if (tmp < 0) {
          continue;
        } else if (tmp == _eccen) {
          _centers.push_back(i);
        } else if (tmp < _eccen) {
          _eccen = tmp;
          _centers.resize(0);
          _centers.push_back(i);
        } else {
          throw std::runtime_error("Logic error in shortest_from");
        }
      }
    }

    void read_excludes(const char *file) {
      std::ifstream fh(file);
      while(not fh.eof()) {
        if (fh.peek() == '#') {
          std::string linebuff;
          std::getline(fh,linebuff);
        }
        int id;
        fh >> id;
        fh >> std::ws;
        if (_nodes->index_of(id) == _nodes->size()) continue;
        if (std::find(_excludes.begin(),_excludes.end(),_nodes->index_of(id)) != _excludes.end()) continue;
        _excludes.push_back(_nodes->index_of(id));
      }
      fh.close();
//    std::cout<<"nb exclude: "<<_excludes.size()<<std::endl;
    }

    int shortest_from(size_t id,int best = -1) {
      if (best < 0) best = _revnodes->size();
      _distances.resize(_revnodes->size());
      _nb_bydist.resize(0);
      for (size_t i = 0; i < _distances.size(); ++i) {_distances[i] = -1;}
      _distances[id] = 0;
      _nb_bydist.push_back(1); // 1 element at distance 0
      std::stack<size_t> queue, incqueue;
      int cdist = 1; // current distance 
      size_t nb_visited = 1; // stop when it reach the total number
      int cfilled = 0; // number of nodes at given distance
      // remove selected
      for (size_t i = 0; i < _excludes.size(); ++i) {
        if (id == _excludes[i]) return -1;
        _distances[_excludes[i]] = 0;
        ++nb_visited;
      }
      queue.push(id);

      filldist:
      {
      const Custom_set & edges = _revnodes->edges_of(queue.top());
      for (size_t i = 0; i < edges.size();++i) {
        if(_distances[edges[i]] < 0) {
          incqueue.push(edges[i]); // add to the list of element to process 
          _distances[edges[i]] = cdist;
          ++nb_visited;
          ++cfilled;
          if (nb_visited == _distances.size()) {
            _nb_bydist.push_back(cfilled);
            return cdist; 
          }
        }
      }
      }
      queue.pop(); // node processed
      if (not queue.empty()) goto filldist;
      // queue is empty, prepare next search
      _nb_bydist.push_back(cfilled);
      cfilled = 0;
      ++cdist;
      if (cdist > best) return -1; // giveup 
      if (incqueue.empty()) throw std::runtime_error("Expected fully connected graph");
      queue.swap(incqueue);
      goto filldist;
    }

    void find_long_path_from(std::vector<int> &out,int lenght) const {
      if (lenght == 0) return;
      const Custom_set & edges = _nodes->edges_of(out.back());
      for (size_t i = 0; i < edges.size(); ++i) {
        if (_distances[edges[i]] == lenght - 1) {
          out.push_back(edges[i]);
          return find_long_path_from(out, lenght - 1);
        }
      }
      throw std::runtime_error("find_long_path: Wrong data");
    }
  
  private:
    const Const_Nodes *_nodes,*_revnodes;
    std::vector<int> _excludes;
    std::vector<int> _distances;
    std::vector<int> _nb_bydist;
    std::vector<int> _centers;
    int _eccen;
};

#endif

