#ifndef TARJAN_HPP_INCLUDED
#define TARJAN_HPP_INCLUDED

#include "const_nodes.hpp"
#include <stack>

#include <map> // for print stats, not in logic

struct vertex_data {
  int index = -1;
  int lowlink = -1;
  int SCC_id = -1;
  bool onStack = false;
};

class Tarjan_SCC {
  public:

    void print_stats(std::ostream &out) {
      out<<"Number of strongly connected components: "<<_SCC_found<<std::endl;
      out<<"Size of each (SCC with more than 10 elements):"<<std::endl;
      std::map<size_t,size_t> nb_bysize;
      for (size_t i = 0; i < _SCC_found; ++i) {
        if (nb_bysize.contains(_SCC[i].size())) {
          nb_bysize[_SCC[i].size()] = nb_bysize[_SCC[i].size()] + 1;
        } else {
          nb_bysize[_SCC[i].size()] = 1;
        }
        if (_SCC[i].size() > 10) 
          out<<i<<":\t"<<_SCC[i].size()<<std::endl;
        else if (_SCC[i].size() == 0)
          throw std::runtime_error("Algorithm failed");
      }
      out<<"Number by size:"<<std::endl;
      for (auto itt = nb_bysize.cbegin(); itt != nb_bysize.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<"\n";
      }
    }

    void print_shortest_path_stats(std::ostream &out) {
      std::map<size_t,int> path_lght_from,path_lght_to;
      for (size_t i = 0; i < _main_index; ++i) {
        if (path_lght_from.contains(_rgraph_shortest[i].size())) {
          path_lght_from[_rgraph_shortest[i].size()] = path_lght_from[_rgraph_shortest[i].size()] + 1;
        } else {
          path_lght_from[_rgraph_shortest[i].size()] = 1;
        }
      }
      for (size_t i = _main_index + 1; i < _rgraph.size() ; ++i) {
        if (path_lght_to.contains(_rgraph_shortest[i].size())) {
          path_lght_to[_rgraph_shortest[i].size()] = path_lght_to[_rgraph_shortest[i].size()] + 1;
        } else {
          path_lght_to[_rgraph_shortest[i].size()] = 1;
        }
      }
      out<<"Shortest path connecting main: "<<std::endl;
      for (auto itt = path_lght_to.cbegin(); itt != path_lght_to.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }
      out<<"Shortest path from main: "<<std::endl;
      for (auto itt = path_lght_from.cbegin(); itt != path_lght_from.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }

    }
    void print_shortest_path_data(std::ostream &out) {
      for (size_t i = 0; i < _rgraph_shortest.size(); ++i) {
        if (_rgraph_shortest[i].size() == 0) {
          out<<i<<": -1\n";
        } else if (_rgraph_shortest[i].size() == 1) {
          out<<i<<": \n";
        } else {
          out<<i<<":";
          if (i < _main_index) {
            for (int j = _rgraph_shortest[i].size() - 1; j > 0; --j){
              out<<" <-- "<<_rgraph_shortest[i][j];
            }
          } else {
            for (int j = _rgraph_shortest[i].size() - 1; j > 0; --j){
              out<<" --> "<<_rgraph_shortest[i][j];
            }
          }
          out<<" ("<<_rgraph_shortest[i].size()<<")\n";
        }
      }
    }

    void print_rgraph_stats(std::ostream &out) {
      int nb_dis = 0, nb_co = 0, nb_by = 0, nb_both = 0, nb_root = 0, max_lght = 0; // also, could just be summed from the other two
      int nb_dis_a = 0, nb_co_a = 0, nb_by_a = 0, nb_both_a = 0, nb_root_a = 0, max_lght_a = 0;
      int nb_dis_b = 0, nb_co_b = 0, nb_by_b = 0, nb_both_b = 0, nb_root_b = 0, max_lght_b = 0;
      std::map<size_t,int> root_lght, root_lght_a, root_lght_b, root_main;
      std::vector<int> max_lght_root_a,max_lght_root_b;
      // specifics datas
      std::map<size_t,int> linked_from_groupe_with_4;
      std::map<size_t,std::vector<int>> long_paths;
      for (size_t i = 0; i < _main_index; ++i) {
        if (_rgraph_connectmain[i] == 3) {++nb_both;++nb_both_b;}
        if (_rgraph_connectmain[i] & 1) {++nb_co;++nb_co_b;}
        else {++nb_dis;++nb_dis_b;}
        if (_rgraph_connectmain[i] & 2) {++nb_by;++nb_by_b;}
        if (_rgraph_isroot[i]) {
          ++nb_root; ++nb_root_b;
          if (root_lght.contains(_rgraph_lght[i])) {
            root_lght[_rgraph_lght[i]] = root_lght[_rgraph_lght[i]] + 1;
          } else {
            root_lght[_rgraph_lght[i]] = 1;
          }
          if (root_lght_b.contains(_rgraph_lght[i])) {
            root_lght_b[_rgraph_lght[i]] = root_lght_b[_rgraph_lght[i]] + 1;
          } else {
            root_lght_b[_rgraph_lght[i]] = 1;
          }
        }
        if (_rgraph_lght[i] > max_lght) {
          max_lght = _rgraph_lght[i];
        }
        if (_rgraph_lght[i] > max_lght_b) {
          max_lght_b = _rgraph_lght[i];
          max_lght_root_b.resize(0);
          max_lght_root_b.push_back(i);
        } else if (_rgraph_lght[i] == max_lght_b) {
          max_lght_root_b.push_back(i);
        }
      }
      for (size_t i = _main_index + 1; i < _SCC.size(); ++i) {
        if (_rgraph_connectmain[i] == 3) {++nb_both;++nb_both_a;}
        if (_rgraph_connectmain[i] & 1) {++nb_co;++nb_co_a;}
        else {++nb_dis;++nb_dis_a;}
        if (_rgraph_connectmain[i] & 2) {++nb_by;++nb_by_a;}
        if (_rgraph_isroot[i]) {
          ++nb_root; ++nb_root_a;
          if (root_lght.contains(_rgraph_lght[i])) {
            root_lght[_rgraph_lght[i]] = root_lght[_rgraph_lght[i]] + 1;
          } else {
            root_lght[_rgraph_lght[i]] = 1;
          }
          if (root_lght_a.contains(_rgraph_lght[i])) {
            root_lght_a[_rgraph_lght[i]] = root_lght_a[_rgraph_lght[i]] + 1;
          } else {
            root_lght_a[_rgraph_lght[i]] = 1;
          }
          if (_rgraph_lght[i] > 10) { // arbitrary cutoff
            if (long_paths.contains(_rgraph_lght[i])) {
              long_paths.at(_rgraph_lght[i]).push_back(i);
            } else {
              long_paths[_rgraph_lght[i]] = {(int)i};
            }
          } else if (_rgraph_lght[i] == 4) { // specific to data
            for (size_t j = 0; j < _rgraph[i].size(); ++j) {
              if (_rgraph_lght[_rgraph[i][j]] == 3) {
                if (linked_from_groupe_with_4.contains(_rgraph[i][j])) {
                  linked_from_groupe_with_4[_rgraph[i][j]] = linked_from_groupe_with_4[_rgraph[i][j]] + 1;
                } else {
                  linked_from_groupe_with_4[_rgraph[i][j]] = 1;
                }
              }
            }
          }
        }
        if (_rgraph_lght[i] > max_lght) {
          max_lght = _rgraph_lght[i];
        }
        if (_rgraph_lght[i] > max_lght_a) {
          max_lght_a = _rgraph_lght[i];
          max_lght_root_a.resize(0);
          max_lght_root_a.push_back(i);
        } else if (_rgraph_lght[i] == max_lght_a) {
          max_lght_root_a.push_back(i);
        }
      }
      std::vector<int> paths;

      out<<"Global data:\nMain group id: "<<_main_index<<" with "<<_SCC[_main_index].size()<<" elements"<<std::endl;
      out<<"Number of disconnected components: "<<nb_dis<<"\n bypassing main: "<<nb_by<<"\n connected: "<<nb_co<<"\n both connected and bypassing: "<<nb_both<<std::endl;;
      out<<"Lenght of longest path: "<<max_lght<<", longest from main: "<<_rgraph_lght[_main_index]<<std::endl;
      for (size_t n = 0; n < _rgraph[_main_index].size(); ++n) {
        if (root_main.contains(_rgraph_lght[_rgraph[_main_index][n]])) {
          root_main[_rgraph_lght[_rgraph[_main_index][n]]] = root_main[_rgraph_lght[_rgraph[_main_index][n]]] + 1;
        } else {
          root_main[_rgraph_lght[_rgraph[_main_index][n]]] = 1;
        }
        if (_rgraph_lght[_rgraph[_main_index][n]] == _rgraph_lght[_main_index] - 1){
          paths.resize(0);
          paths.push_back(_main_index);
          paths.push_back(_rgraph[_main_index][n]);
          find_long_path_from(paths,_rgraph_lght[_main_index] - 1);
          if (paths.size() > 1) {
            for (size_t i = 0; i < paths.size() - 1; ++i) {
              out<<paths[i]<<" -> ";
            }
            out<<paths.back()<<std::endl;
          }
        }
      }
      out<<"Longest path from "<<nb_root<<" roots: "<<std::endl;
      for (auto itt = root_lght.cbegin(); itt != root_lght.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }
      out<<"Longest path from main"<<std::endl;
      for (auto itt = root_main.cbegin(); itt != root_main.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }
      out<<"\nData from above main:"<<std::endl;
      out<<"Number of disconnected components: "<<nb_dis_a<<"\n bypassing main: "<<nb_by_a<<"\n connected: "<<nb_co_a<<"\n both connected and bypassing: "<<nb_both_a<<std::endl;;
      out<<"Lenght of longest path: "<<max_lght_a;
      if (max_lght_root_a.size() > 1) {
        out<<" ("<<max_lght_root_a.size()<<" times)"<<std::endl;
      } else {
        out<<std::endl;
      }
      for (size_t n = 0; n < max_lght_root_a.size(); ++n) {
        paths.resize(0);
        paths.push_back(max_lght_root_a[n]);
        find_long_path_from(paths,max_lght_a);
        if (paths.size() > 1) {
          for (size_t i = 0; i < paths.size() - 1; ++i) {
            out<<paths[i]<<" -> ";
          }
          out<<paths.back()<<std::endl;
        }
      }
      out<<"Longest path from "<<nb_root_a<<" roots: "<<std::endl;
      for (auto itt = root_lght_a.cbegin(); itt != root_lght_a.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }
      // More specific data
      out<<"\nOther long paths:"<<std::endl;
      for (auto itt = long_paths.crbegin(); itt != long_paths.crend(); ++itt) {
        if (itt->first == max_lght_a) continue; // already printed
        out<<itt->first<<":"<<std::endl;
        for (size_t n = 0; n < itt->second.size(); ++n) {
          paths.resize(0);
          paths.push_back(itt->second[n]);
          find_long_path_from(paths,itt->first);
          if (paths.size() > 1) {
            for (size_t i = 0; i < paths.size() - 1; ++i) {
              out<<paths[i]<<" -> ";
            }
            out<<paths.back()<<std::endl;
          }
        }
      }
      // Even more specific 
 /*     out<<"Link from root with lenght 4:"<<std::endl;
      for (auto itt = linked_from_groupe_with_4.cbegin(); itt != linked_from_groupe_with_4.cend(); ++itt) {
        out<<itt->first<<":"<<itt->second<<std::endl;
      }*/
      out<<"\nData from below main:"<<std::endl;
      out<<"Number of disconnected components: "<<nb_dis_b<<"\n bypassing main: "<<nb_by_b<<"\n connected: "<<nb_co_b<<"\n both connected and bypassing: "<<nb_both_b<<std::endl;;
      out<<"Lenght of longest path: "<<max_lght_b;
      if (max_lght_root_b.size() > 1) {
        out<<" ("<<max_lght_root_b.size()<<" times)"<<std::endl;
      } else {
        out<<std::endl;
      }
      for (size_t n = 0; n < max_lght_root_b.size(); ++n) {
        paths.resize(0);
        paths.push_back(max_lght_root_b[n]);
        find_long_path_from(paths,max_lght_b);
        if (paths.size() > 1) {
          for (size_t i = 0; i < paths.size() - 1; ++i) {
            out<<paths[i]<<" -> ";
          }
          out<<paths.back()<<std::endl;
        }
      }
      out<<"Longest path from "<<nb_root_b<<" roots: "<<std::endl;
      for (auto itt = root_lght_b.cbegin(); itt != root_lght_b.cend(); ++itt) {
        out<<itt->first<<": "<<itt->second<<std::endl;
      }

    }

    void find_long_path_from(std::vector<int> &out,int lenght) const {
      if (lenght == 0) return;
      const Custom_set & edges = _rgraph[out.back()];
      for (size_t i = 0; i < edges.size(); ++i) {
        if (_rgraph_lght[edges[i]] == lenght - 1) {
          out.push_back(edges[i]);
          return find_long_path_from(out, lenght - 1);
        }
      }
      throw std::runtime_error("find_long_path: Wrong data");
    }

    void print_data(std::ostream &out) {
      out<<"SCC using raw index ([SCC index]\t[size]: list of raw_id), start at 0"<<std::endl;

      for (size_t i = 0; i < _SCC_found; ++i) {
        out<<i<<"\t"<<_SCC[i].size()<<":\n";
        for (size_t j = 0; j < _SCC[i].size(); ++j) {
          out<<_SCC[i][j]<<"\n";
        }
        out<<std::endl;
      }
    }
    void print_rgraph_data(std::ostream &out) {
      out<<"SCC index\tconnected\troot\tlght"<<std::endl;
      for (size_t i = 0; i < _SCC.size(); ++i) {
        out<<i<<"\t"<<_rgraph_connectmain[i]<<"\t"<<_rgraph_isroot[i]<<"\t"<<_rgraph_lght[i]<<std::endl;
      }
    }
    void print_id_outsize_main(std::ostream &out) {
      for (size_t i = 0; i < _vdata.size();++i) {
        if (_vdata[i].SCC_id != _main_index) {
          out<<_nodes->id(i)<<std::endl;
        }
      }
    }

    void compute_SCC(Const_Nodes const *nodes) {
      if (_index != 0) throw std::runtime_error("Tarjan_SCC not reinitialized");
      _vdata.resize(nodes->size());
      _nodes = nodes;
      for (size_t i = 0; i < _vdata.size(); ++i) {
        if (_vdata[i].index < 0) {
          strongconnect(i);
        }
      }
      // get maingroup
      size_t maxs = 0, max_index = 0;
      for (size_t i = 0; i < _SCC.size(); ++i) {
        if (_SCC[i].size() > maxs) {
          maxs = _SCC[i].size();
          max_index = i;
        }
      }
      _main_index = max_index;
    }

    void strongconnect(int ini_index) {
      std::stack<size_t> c_index,last_loop_itt; // recursion overflow, replace with goto
      // c_index contains the current value of v_ind, last_loop_itt the current value of the itterator in the loop over edges
      size_t v_ind, w_ind, edges_itt; // no initializer to allow jumping past it
      c_index.push(ini_index);
    strongconnect_recursive:
      v_ind = c_index.top(); // get the argument 
      { // block to use a reference with goto (the return must be outside)
      vertex_data &v = _vdata[v_ind];
      v.index = _index;
      v.lowlink = _index;
      ++_index;
      _S.push(v_ind); 
      v.onStack = true;
      }
      for (edges_itt = 0; edges_itt < _nodes->edges_of(v_ind).size();++edges_itt) {
        w_ind = _nodes->edges_of(v_ind)[edges_itt];
        if (_vdata[w_ind].index < 0) {
          // call strongconnect
          last_loop_itt.push(edges_itt); // store current itterator value
          c_index.push(w_ind); // push the function argument
          goto strongconnect_recursive; // call strongconnect, this call must return on the next line
          strongconnect_recursivereturn:
          // recover state
          w_ind = c_index.top(); // the function must return the stack in the same state, hence w_ind is at the top 
          c_index.pop(); // next value is v_ind
          v_ind = c_index.top();
          edges_itt = last_loop_itt.top(); 
          last_loop_itt.pop(); 
          _vdata[v_ind].lowlink = std::min(_vdata[v_ind].lowlink, _vdata[w_ind].lowlink);
        } else if (_vdata[w_ind].onStack) { // w is in the current SCC, else it is in one already found
          _vdata[v_ind].lowlink = std::min(_vdata[v_ind].lowlink, _vdata[w_ind].index);
        }
      }
      // if with init_statement
      if (vertex_data &v = _vdata[v_ind]; v.lowlink == v.index) { // v is the root of a SCC
        _SCC.emplace_back();
        for (size_t w_ind = _S.top(); w_ind != v_ind; w_ind = _S.top()) {
          _S.pop();
          _vdata[w_ind].onStack = false;
          _vdata[w_ind].SCC_id = _SCC_found;
          _SCC[_SCC_found].push_back(_nodes->id(w_ind));
        }
        _S.pop(); // remove v_ind from stack
        _vdata[v_ind].onStack = false;
        _vdata[v_ind].SCC_id = _SCC_found;
        _SCC[_SCC_found].push_back(_nodes->id(v_ind));
        ++_SCC_found;
      }
      // end of the function, if the stack last_loop_itt is empty then it was called from outside, else it was called from the inner loop
      if (last_loop_itt.empty()) return;
      goto strongconnect_recursivereturn;
    }

    void compute_reduced_graph(Const_Nodes const *rev_nodes) {
      _rgraph.resize(_SCC.size());
      for (size_t i = 0; i < _vdata.size(); ++i) { // no group can link to group of higher order
        if (_vdata[i].SCC_id < _main_index) { // use reverse data to avoid parsing 6M elements
          const Custom_set & edges = rev_nodes->edges_of(i);
          for (size_t j = 0; j < edges.size(); ++j) {
            if (_vdata[edges[j]].SCC_id != _vdata[i].SCC_id) // discard loop 
                _rgraph[_vdata[edges[j]].SCC_id].add_unordered(_vdata[i].SCC_id);
          }
        } else if (_vdata[i].SCC_id > _main_index) {
          const Custom_set & edges = _nodes->edges_of(i);
          for (size_t j = 0; j < edges.size(); ++j) {
            if (_vdata[edges[j]].SCC_id != _vdata[i].SCC_id) // discard loop 
                _rgraph[_vdata[i].SCC_id].add_unordered(_vdata[edges[j]].SCC_id);
          }
        } // if
      } // for all edges
    }
    
    void compute_path_lenght() { // the reduced graph is topologically ordered directed acyclic
      _rgraph_lght.resize(_rgraph.size());
      for (size_t i = 0; i < _rgraph_lght.size(); ++i) {
        _rgraph_lght[i] = -1;
      }
      for (int i = _rgraph_lght.size() -1 ; i >= 0; --i) {
        if (_rgraph_lght[i] < 0) {
          path_lenght(i);
        }
      }
    }

    void path_lenght(size_t index) { // must only be called on vertices not yet visited
      std::stack<size_t> S_index;
      S_index.push(index);
      path_lenght_start:
      int lght = 0;
      Custom_set & edges = _rgraph[S_index.top()];
      for (size_t i = 0; i < edges.size(); ++i) {
        if (_rgraph_lght[edges[i]] < 0) { // redo the function until we find a vertex without outgoing arrows
          S_index.push(edges[i]);
          goto path_lenght_start;
        }
        lght = std::max(_rgraph_lght[edges[i]] + 1,lght);
      }
      _rgraph_lght[S_index.top()] = lght;
      S_index.pop();
      if (S_index.empty()) return;
      goto path_lenght_start;
    }
    
    void compute_connection_main() {
      _rgraph_connectmain.resize(_rgraph.size());
      _rgraph_isroot.resize(_rgraph.size());
      for (size_t i = 0; i < _rgraph_connectmain.size(); ++i) {
        _rgraph_connectmain[i] = -1;
        _rgraph_isroot[i] = 0;
      }
      _rgraph_connectmain[_main_index] = 1;
      for (int i = _rgraph_connectmain.size() - 1; i > _main_index; --i) {
        if (_rgraph_connectmain[i] < 0) {
          connect_main_above(i);
          _rgraph_isroot[i] = 1; // was not selected before
        }
      }
      const Custom_set &edges = _rgraph[_main_index]; 
      for (size_t i = 0; i < edges.size(); ++i) {
        if (_rgraph_connectmain[edges[i]] < 0) {
          _rgraph_connectmain[edges[i]] = 1; // connected to main but not to any other before
        } else { // it must be 2 
          if (_rgraph_connectmain[edges[i]] != 2) throw std::runtime_error("Connect main logic failed"); // DEBUG
          _rgraph_connectmain[edges[i]] = 3; // connected to both
        }
      }
      for (int i = _main_index - 1; i >= 0; --i) {
        if (_rgraph_connectmain[i] < 0) {
          _rgraph_isroot[i] = 1; 
          _rgraph_connectmain[i] = 0; // cannot be reach anymore
        } 
        connect_main_below(i);
      }
    }

    int connect_main_above(size_t index) {
      if (_rgraph_connectmain[index] < 0) {
        const Custom_set & edges = _rgraph[index];
        _rgraph_connectmain[index] = 0;
        for (int i = edges.size() - 1; i >= 0; --i) {
          if (edges[i] > _main_index) {
            _rgraph_connectmain[index] |= connect_main_above(edges[i]);
          } else if (edges[i] == _main_index) {
            _rgraph_connectmain[index] |= 1; // connect main
          } else {
            _rgraph_connectmain[index] |= 2; // connected to one after main
            _rgraph_connectmain[edges[i]] = 2; // directly connect before main
          }
        }
      }
      return _rgraph_connectmain[index];
    }

    void connect_main_below(size_t index) {
      const Custom_set & edges = _rgraph[index];
      for (int i = edges.size() - 1; i >= 0; --i) { // propagate down the properties
        if (_rgraph_connectmain[edges[i]] < 0) {
          _rgraph_connectmain[edges[i]] = _rgraph_connectmain[index]; // never visited, add initial value
        } else {
          _rgraph_connectmain[edges[i]] |= _rgraph_connectmain[index]; // add value
        }
      }
    }

    
    void compute_shortest_path(std::ostream &out) { // the reduced graph is topologically ordered directed acyclic
      // compute reverse graph
      out<<"Reversing..."<<std::flush;
      _rgraph_rev.resize(_rgraph.size());
      for (size_t i = 0; i < _rgraph.size();++i) {
        for(size_t j = 0; j < _rgraph[i].size();++j) {
          _rgraph_rev[_rgraph[i][j]].add_ordered(i);
        }
      }
      _rgraph_shortest.resize(_rgraph.size());
      _shortest_to_main(_main_index); // fill path for nodes directly connected
      for (size_t i = _main_index + 1; i < _rgraph_shortest.size(); ++i) {
        out<<"\r"<<i<<std::flush;
        if (_rgraph_shortest[i].size() > 0) { // if it is already connected
          _shortest_to_main(i); // try to connect other 
        }
      }
      // lower
      out<<"\r       "<<std::flush;
      _shortest_from_main(_main_index);
      for (int i = _main_index - 1; i >= 0; --i) {
        out<<"\r"<<i<<std::flush;
        _shortest_from_main(i);
      }
      out<<"\r       \r"<<std::flush;
    }

    void _shortest_to_main(size_t index) { // for above, call on index from nodes already connected
      for(size_t i = 0; i < _rgraph_rev[index].size();++i){
        if (_rgraph_shortest[_rgraph_rev[index][i]].size() == 0 || _rgraph_shortest[_rgraph_rev[index][i]].size() > _rgraph_shortest[index].size() + 1) {
          _rgraph_shortest[_rgraph_rev[index][i]] = _rgraph_shortest[index];
          _rgraph_shortest[_rgraph_rev[index][i]].push_back(index);
        }
      }
    }
    void _shortest_from_main(size_t index) { // for below
      for (size_t i = 0; i < _rgraph[index].size(); ++i) {
        if (_rgraph_shortest[_rgraph[index][i]].size() == 0 || _rgraph_shortest[_rgraph[index][i]].size() > _rgraph_shortest[index].size() + 1) {
          _rgraph_shortest[_rgraph[index][i]] = _rgraph_shortest[index];
          _rgraph_shortest[_rgraph[index][i]].push_back(index);
        }
      }
    }

  private:
    std::vector<int> _rgraph_connectmain; // 0 do not connect, 1 connect, 2 bypass, 3 both
    std::vector<int> _rgraph_isroot;
    std::vector<int> _rgraph_lght;
    std::vector<std::vector<int>> _rgraph_shortest;
    std::vector<Custom_set> _rgraph; // use index of SCC
    std::vector<Custom_set> _rgraph_rev; // use index of SCC
    size_t _main_index;
    const Const_Nodes * _nodes;
    std::vector<vertex_data> _vdata;
    std::stack<size_t> _S;
    int _index = 0;
    int _SCC_found = 0;
    std::vector<std::vector<int>> _SCC;
};

#endif

